﻿using System.Collections;
using System.Collections.Generic;
using System.Data;
using UnityEngine;
using UnityEngine.EventSystems;
using UnityEngine.UI;

//CustomButton internally manages a Unity button but adds custom logic and properties to it
public class CustomButton : MonoBehaviour, IPointerEnterHandler, IPointerExitHandler, IPointerDownHandler, IPointerUpHandler
{
    //Public properties
    public Button Button { get; private set; }
    public bool IsPressed { get; private set; }
    public bool IsBackButton = false;

    private Animator _Animator;

    /// <summary>
    /// Enables the button, playing the correct animation
    /// </summary>
    public void Enable()
    {
        Button = GetComponent<Button>();
        Button.interactable = true;

        if (_Animator != null)
        {
            _Animator?.SetBool("IsEnabled", true);
        }
    }

    /// <summary>
    /// Disables the button, playing the correct animation
    /// </summary>
    public void Disable()
    {
        Button = GetComponent<Button>();
        Button.interactable = false;

        if (_Animator != null)
        {
            _Animator?.SetBool("IsEnabled", false);
        }
    }

    void Awake()
    {
        //Get our components
        Button = GetComponent<Button>();
        _Animator = GetComponent<Animator>();
        gameObject.AddComponent<AudioSource>();

        //Play the correct animation and set the state depending on the initial property values
        if (Button.IsInteractable())
        {
            if (_Animator != null)
            {
                _Animator.Play("OnUnfocused");
            }

            Enable();
        }

        else
        {
            if (_Animator != null)
            {
                _Animator.Play("OnInactiveUnfocused");
            }

            Disable();
        }
    }

    /// <summary>
    /// Handler for when the pointer enters the button to focus on the button
    /// </summary>
    /// <param name="eventData"></param>
    public void OnPointerEnter(PointerEventData eventData)
    {
        OnFocus();
    }

    /// <summary>
    /// Handler for when the button is clicked
    /// </summary>
    public void OnClick()
    {
        if (Button.IsInteractable())
        {
            //Play the correct sound file and the animation
            if (!IsBackButton)
            {
                AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles[Constants.UIButtonSelectAudioID]);
            }

            else
            {
                AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles[Constants.UIButtonBackAudioID]);
            }

            if (_Animator != null)
            {
                _Animator.Play("OnPressed");
            }
        }

        else
        {
            //Play the sound file and the animation
            AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles[Constants.UIButtonInactiveSelectAudioID]);

            if (_Animator != null)
            {
                _Animator.Play("OnInactivePressed");
            }
        }
    }

    /// <summary>
    /// Handler for when the pointer leaves the button to unfocus on the button
    /// </summary>
    /// <param name="eventData"></param>
    public void OnPointerExit(PointerEventData eventData)
    {
        OnLostFocus();
    }

    /// <summary>
    /// Sets up the button for the focused state
    /// </summary>
    private void OnFocus()
    {
        AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles[Constants.UIButtonFocusAudioID]);  //Play the audio

        if (_Animator != null)
        {
            //Play the correct animation for the button state
            if (Button.IsInteractable())
            {
                _Animator.Play("OnFocusBegin");
            }

            else
            {
                _Animator.Play("OnInactiveFocusBegin");
            }
        }
    }

    /// <summary>
    /// Sets up the button for the unfocused state
    /// </summary>
    private void OnLostFocus()
    {
        if (_Animator != null)
        {
            //Play the correct animation for the button state
            if (Button.IsInteractable())
            {
                _Animator.Play("OnLostFocus");
            }

            else
            {
                _Animator.Play("OnInactiveLostFocus");
            }
        }
    }

    /// <summary>
    /// Sets the button to be pressed
    /// </summary>
    /// <param name="eventData"></param>
    public void OnPointerDown(PointerEventData eventData)
    {
        OnClick();
        IsPressed = true;
    }

    /// <summary>
    /// Sets the button to be unpressed 
    /// </summary>
    /// <param name="eventData"></param>
    public void OnPointerUp(PointerEventData eventData)
    {
        IsPressed = false;
    }
}
